import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import seaborn as sns
from pymongo import MongoClient
from pandas.io.json import json_normalize
from pylab import mpl
# 导入 pylab 模块中的 mpl(matplotlib 的配置接口), 用于深度定制 matplotlib 的全局参数(如字体、符号等)

plt.style.use('ggplot')
# 设置 matplotlib 的绘图风格为ggplot
mpl.rcParams['font.sans-serif'] = ['SimHei']
# 设置 matplotlib 的默认无衬线字体为 'SimHei'黑体，解决中文显示为方框的问题
# 'font.sans-serif': 指定无衬线字体列表（多个字体按优先级排列）
plt.rc('figure', figsize=(10, 10))
# 设置所有 matplotlib 图表的默认大小为 10x10 英寸, 把plt默认的图片size调大一点
# 'figure': 指定配置对象为图表; figsize=(width, height): 宽和高（单位：英寸）
plt.rcParams["figure.dpi"] = 100
# 设置图表的默认分辨率为 100 DPI(每英寸像素数)
mpl.rcParams['axes.unicode_minus'] = False
# 禁用 Unicode 的负号显示, 解决保存图像是负号'-'显示为方块的问题
%matplotlib inline
# Jupyter Notebook 的魔法命令, 使 matplotlib 图表直接显示在单元格下方


conn = MongoClient(host='127.0.0.1', port=27017)
# 实例化MongoClient. conn是MongoDB 连接对象
db = conn.get_database('Lianjia')
# 连接到Lianjia数据库. db是数据库对象, 操作指定的 MongoDB 数据库
zufang = db.get_collection('zufang')
# 连接到集合zufang
mon_data = zufang.find()
# mon_data是查询游标, 查询这个集合下的所有记录. 它本身不直接存储数据, 而是像一个“懒加载迭代器”, 只有在实际遍历时才会从数据库逐条提取数据
data = json_normalize([comment for comment in mon_data])
# data是Pandas DataFrame, 将 MongoDB 的 JSON 数据转为表格形式，便于分析
# json_normalize 专门用于将嵌套的 JSON 数据(或字典列表)转换为扁平化的 Pandas DataFrame. 它非常适合处理 MongoDB 返回的嵌套文档.
# 每一列的名字在实施后为 _id, bathroom_num, bedroom_num, bizcircle_name, city, dist, distance, frame_orientation, hall_num, house_tag, house_title, latitude, layout, longitud, m_url, rent_area, rent_price_listing, rent_price_unit, resblock_name, type



data_sample = pd.concat([data[data['city']==city].sample(3000) for city in ['北京', '上海', '广州', '深圳']])
# 每个城市各采样3000条数据, 将 4 个城市的样本数据纵向拼接成一个大的 DataFrame
data_sample.to_csv('data_sample.csv', index=False)
# 将 DataFrame 保存为 CSV 文件, 路径为'data_sample.csv'
# index = False: 不将 DataFrame 的索引列写入文件



# 数据清洗(按列清理)
# 1. 去掉“_id”列
data = data.drop(columns='_id')
# 2. 通过每一列判断有没有异常元素
# print(data[Column_Name].unique())
# data = data.drop([Index])
# 3. rent_area字段有些填写的是一个范围，比如23-25平房米，后期转换成“float”类型的时候不好转换，考虑取平均值; 价格同理
def get_aver(data):
    if isinstance(data, str) and '-' in data:
        low, high = data.split('-')
        return (int(low)+int(high))/2
    else:
        return int(data)
data['rent_area'] = data['rent_area'].apply(get_aver)
data['rent_price_listing'] = data['rent_price_listing'].apply(get_aver)
# 4. 数据类型转换. 'distance', 'latitude', 'longitude'因为有None, 需另外处理
for col in ['bathroom_num', 'bedroom_num', 'hall_num', 'rent_price_listing']:
    data[col] = data[col].astype(int)
def dw_None_dis(data):
    if data is None:
        return np.nan
    else:
        return int(data)
def dw_None_latlon(data):
    if data is None or data == '':
        return np.nan
    else:
        return float(data)            
data['distance'] = data['distance'].apply(dw_None_dis)
data['latitude'] = data['latitude'].apply(dw_None_latlon)
data['longitude'] = data['longitude'].apply(dw_None_latlon)
# 5. 将清理好的数据打包
data.to_csv('data_clean.csv', index=False)



### 1. 各城市的租房分布怎么样？ ###



def get_city_zf_loc(city, city_short, col=['longitude', 'latitude', 'dist'], data=data):
# col: 需要保留的列(默认包含经度、纬度和行政区). data: 输入的DataFrame(默认使用全局变量data)
    file_name = 'data_' + city_short + '_latlon.csv'
    data_latlon = data.loc[data['city']==city, col].dropna(subset=['latitude', 'longitude'])
    # data.loc[data['city']==city, col]: 筛选指定城市的数据, 仅保留col列(默认经度、纬度、行政区)
    data_latlon['longitude'] = data_latlon['longitude'].astype(str)
    data_latlon['latitude'] = data_latlon['latitude'].astype(str)
    # 将经度和纬度列从任意类型（如float）转为字符串类型，为后续拼接做准备
    data_latlon['latlon'] = data_latlon['longitude'].str.cat(data_latlon['latitude'], sep=',')
    # str.cat(): 将经度和纬度列用逗号拼接, 生成新列latlon. 参数sep=','指定分隔符
    data_latlon.to_csv(file_name, index=False)
    # 数据打包
    print(city+'的数据一共有{}条'.format(data_latlon.shape[0]))
get_city_zf_loc('北京', 'bj', ['longitude','latitude', 'dist'])
get_city_zf_loc('上海', 'sh', ['longitude','latitude', 'dist'])
get_city_zf_loc('广州', 'gz', ['longitude','latitude', 'dist'])
get_city_zf_loc('深圳', 'sz', ['longitude','latitude', 'dist'])
fig = plt.figure(dpi=300)
# 创建一个新的图形对象. 设置分辨率为300
data.dropna(subset=['latitude', 'longitude'])[data['city']=='北京']['dist'].value_counts(ascending=True).plot.barh()
# fig = plt.figure(dpi=300) 会初始化一个空画布, 并设置分辨率, 但此时画布上没有任何内容. 当执行后续的 plot.barh() 时, Matplotlib 会 自动将图形绘制到当前活动的 Figure 对象(即刚创建的 fig)上.
# data.dropna(subset=['latitude', 'longitude']): 删除经纬度缺失值
# .value_counts(ascending=True): 统计各行政区房源数(升序)
# .plot.barh(): 绘制水平条形图



### 2. 城市各区域的房价分布怎么样？ ###



data['aver_price'] = np.round(data['rent_price_listing'] / data['rent_area'], 1)
# np.round(x, 1): 四舍五入保留1位小数
g = sns.FacetGrid(data, row="city", height=4, aspect=2)
g = g.map(sns.kdeplot, "aver_price")
# 分城市绘制租金分布密度图
# sns.FacetGrid(): 创建分面网格，用于多子图绘制. row="city":按城市分行. height=4:每子图高度4英寸. aspect=2:宽高比2(宽度=高度×2)
# .map(sns.kdeplot, "aver_price"):在每个子图绘制核密度估计(KDE)曲线

# 由于平均租金基本上都集中在250元/平米/月以内，所以选取这部分数据绘制热力图
def get_city_zf_aver_price(city, city_short, col=['longitude', 'latitude', 'aver_price'], data=data):
    file_name = 'data_' + city_short + '_aver_price.csv'
    data_latlon = data.loc[(data['city']==city)&(data['aver_price']<=250), col].dropna(subset=['latitude', 'longitude'])
    data_latlon['longitude'] = data_latlon['longitude'].astype(str)
    data_latlon['latitude'] = data_latlon['latitude'].astype(str)
    data_latlon['latlon'] = data_latlon['longitude'].str.cat(data_latlon['latitude'], sep=',')
    data_latlon.to_csv(file_name, index=False)
    print(city+'的数据一共有{}条'.format(data_latlon.shape[0]))
get_city_zf_aver_price('北京', 'bj')
get_city_zf_aver_price('上海', 'sh')
get_city_zf_aver_price('广州', 'gz')
get_city_zf_aver_price('深圳', 'sz')

# 各城市租金Top10的商圈
bc_top10 = data.groupby(['city', 'bizcircle_name'])['aver_price'].mean().nlargest(50).reset_index()['city'].value_counts()
from pyecharts import Bar
# groupby(['city', 'bizcircle_name']): 按城市和商圈分组; ['aver_price'].mean(): 计算每组平均单价; .nlargest(50): 取单价最高的50个商圈; .reset_index(): 将分组索引还原为普通列; ['city'].value_counts(): 统计各城市在Top50中出现的次数
bar = Bar("每平米平均租金前50的北上广深商圈数量", width=400)
# Bar(title, width): 创建柱状图对象
bar.add("", bc_top10.index, bc_top10.values, is_stack=True,xaxis_label_textsize=16, yaxis_label_textsize=16, is_label_show=True)
# .add(series_name, x_axis, y_axis, **kwargs): 添加数据
# is_stack=True: 堆叠模式(此处实际无用，可移除)
# is_label_show=True: 显示柱顶数值

def get_top10_bc(city, data=data):
    top10_bc = data[(data['city']==city)&(data['bizcircle_name']!='')].groupby('bizcircle_name')['aver_price'].mean().nlargest(10)
    bar = Bar(city+"市每平米平均租金Top10的商圈", width=600)
    bar.add("", top10_bc.index, np.round(top10_bc.values, 0), is_stack=True,xaxis_label_textsize=16, yaxis_label_textsize=16, xaxis_rotate=30, is_label_show=True)
    return bar
# 北京每平米平均租金Top10的商圈
get_top10_bc('北京')



### 3. 距离地铁口远近有什么关系？ ###



from scipy import stats
def distance_price_relation(city, data=data):
    g = sns.jointplot(x="distance", 
                  y="aver_price", 
                  data=data[(data['city']==city)&
                            (data['aver_price']<=350)].dropna(subset=['distance']), 
                  kind="reg",
                 stat_func=stats.pearsonr)
    g.fig.set_dpi(100)
    g.ax_joint.set_xlabel('最近地铁距离', fontweight='bold')
    g.ax_joint.set_ylabel('每平米租金', fontweight='bold')
    return g
distance_price_relation('北京')
bins = [100*i for i in range(13)]
data['bin'] = pd.cut(data.dropna(subset=['distance'])['distance'], bins)
bin_bj = data[data['city']=='北京'].groupby('bin')['aver_price'].mean()
bin_sh = data[data['city']=='上海'].groupby('bin')['aver_price'].mean()
bin_gz = data[data['city']=='广州'].groupby('bin')['aver_price'].mean()
bin_sz = data[data['city']=='深圳'].groupby('bin')['aver_price'].mean()
from pyecharts import Line

line = Line("距离地铁远近跟每平米租金均价的关系")
for city, bin_data in {'北京':bin_bj, '上海':bin_sh, '广州':bin_gz, '深圳':bin_sz}.items():
    line.add(city, bin_data.index, bin_data.values,
            legend_text_size=18,xaxis_label_textsize=14,yaxis_label_textsize=18,
             xaxis_rotate=20, yaxis_min=8, legend_top=30)
line
